package org.dyndns.kwitte.smartcrypt.ui.controller;

import javax.swing.*;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.datatransfer.Clipboard;
import java.awt.event.InputEvent;
import java.io.IOException;
import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

/**
 * A class to handle dropped files additionally to the default behaviour
 * of <code>TransferHandler</code>. {@link #dropFiles} must be overridden
 * to do with the dropped files whatever you want.
 * All methods must be called in the EDT, and the instance must be created
 * in the EDT.
 */
public abstract class DropFileHandler extends TransferHandler {
    private TransferHandler wrappedHandler;

    protected DropFileHandler() {
    }

    /**
     * Creates a new instance. The behaviour of this class,
     * <code>TransferHandler</code> and <code>handler</code> is added.
     * For each event all three <code>TransferHandler</code>s get a chance
     * to handle the event in an unspecified order.
     * <strong><code>createTransferable</code> and <code>exportDone</code>
     * however are not called on the wrapped handler!</strong>
     * @param handler the <code>TransferHandler</code> to be wrapped
     */
    protected DropFileHandler(TransferHandler handler) {
        this.wrappedHandler = handler;
    }

    /**
     * This method is called when a file is being dropped.
     * @param f an array of <code>File</code>s which have been dropped
     */
    public abstract void dropFiles(File[] f);

    @Override
    public boolean importData(JComponent comp, Transferable t) {
        boolean importSuccess = false;

        try {
            File[] f = grepFiles(t);
            if (f != null) {
                dropFiles(f);
                importSuccess = true;
            }
        } catch (UnsupportedFlavorException e) {
            throw new AssertionError(e);
        } catch (IOException e) {
            // reason: the data is no longer available in the requested flavor
            Logger.getLogger("org.dyndns.kwitte.smartcrypt.ui.controller")
                .throwing("DropFileHandler", "importData", e);
        }

        // give "everyone" a chance to handle the import (super, wrapped)
        return super.importData(comp, t)
               | (wrappedHandler != null && wrappedHandler.importData(comp, t))
               | importSuccess;
    }

    private File[] grepFiles(Transferable t)
            throws UnsupportedFlavorException, IOException {
        for (DataFlavor flavor : t.getTransferDataFlavors()) {
            if (flavor.equals(DataFlavor.javaFileListFlavor)) {
                return (File[]) ((List) t.getTransferData(flavor))
                    .toArray(new File[0]);
            }
        }
        return null;
    }

    @Override
    public boolean canImport(JComponent comp, DataFlavor[] transfers) {
        // true iff any of the three can handle it (this, wrapped, superclass)
        return
            Arrays.asList(transfers).contains(DataFlavor.javaFileListFlavor)
            || (wrappedHandler != null
                && wrappedHandler.canImport(comp, transfers))
            || super.canImport(comp, transfers);
    }

    @Override public void exportAsDrag(JComponent comp, InputEvent e,
                                       int action) {
        super.exportAsDrag(comp, e, action);
        if (wrappedHandler != null) {
            wrappedHandler.exportAsDrag(comp, e, action);
        }
    }

    @Override public void exportToClipboard(JComponent comp, Clipboard clip,
                                            int action)
        throws IllegalStateException {
        super.exportToClipboard(comp, clip, action);
        if (wrappedHandler != null) {
            wrappedHandler.exportToClipboard(comp, clip, action);
        }
    }

    @Override public int getSourceActions(JComponent c) {
        int ret = super.getSourceActions(c);
        if (ret == NONE && wrappedHandler != null) {
            ret = wrappedHandler.getSourceActions(c);
        }
        return ret;
    }

    @Override public Icon getVisualRepresentation(Transferable t) {
        Icon ret = super.getVisualRepresentation(t);
        if (ret == null && wrappedHandler != null) {
            ret = wrappedHandler.getVisualRepresentation(t);
        }
        return ret;
    }

    @Override protected Transferable createTransferable(JComponent c) {
        return super.createTransferable(c);
    }

    @Override protected void exportDone(JComponent source, Transferable data,
                                        int action) {
        super.exportDone(source, data, action);
    }
}
