/*
 * Smartcrypt - a tool to encrypt / decrypt files
 * Copyright (C) 2005 Kai Witte
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * For contact information see http://kwitte.dyndns.org/~kwitte/
 */
package org.dyndns.kwitte.smartcrypt.ui.controller;

import org.dyndns.kwitte.smartcrypt.*;
import org.dyndns.kwitte.smartcrypt.ui.ProgressDialog;
import org.dyndns.kwitte.smartcrypt.ui.MainView;
import org.dyndns.kwitte.smartcrypt.ui.ConfirmDialog;

import javax.swing.*;
import javax.crypto.Cipher;
import java.awt.event.ActionEvent;
import java.awt.*;
import java.security.NoSuchAlgorithmException;
import java.security.InvalidKeyException;
import java.util.logging.Logger;
import java.util.ResourceBundle;
import java.util.Locale;
import java.io.IOException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;

/**
 * An action to trigger encryption or decryption.
 */
public class CryptAction extends AbstractAction {
    private ResourceBundle resourceBundle = ResourceBundle.getBundle(
        "org/dyndns/kwitte/smartcrypt/ui/mainview"
    );
    private final int mode;
    private volatile boolean cancelled;
    private volatile boolean overwriteAllWithoutConfirmation;
    private volatile boolean neverOverwriteExistingFile;
    private boolean confirmPassword;
    private static final String FILE_SEPARATOR =
        System.getProperty("file.separator");

    /**
     * Creates a new instance.
     * @param mode either <code>Cipher.ENCRYPT_MODE</code> or
     * <code>Cipher.DECRYPT_MODE</code>
     * @param confirmPassword true iff the password shall be confirmed
     * @throws IllegalArgumentException iff the mode value is illegal
     */
    public CryptAction(int mode, boolean confirmPassword) {
        this.confirmPassword = confirmPassword;
        if (mode != Cipher.ENCRYPT_MODE && mode != Cipher.DECRYPT_MODE) {
            throw new IllegalArgumentException();
        }
        super.setEnabled(false);
        this.confirmPassword = confirmPassword;
        this.mode = mode;
    }

    /**
     * Triggers encryption.
     * @param e not used.
     */
    public void actionPerformed(ActionEvent e) {
        overwriteAllWithoutConfirmation = false;
        neverOverwriteExistingFile = false;
        cancelled = false;

        final ProgressDialog progress =
            new ProgressDialog(MainView.getInstance());

        final File infile = new File(MainView.getInstance().getInfile());
        final File outfile = new File(MainView.getInstance().getOutfile());
        final String password = MainView.getInstance().getPassword();
        final String confirmedPassword =
            MainView.getInstance().getPasswordConfirm();
        final String algorithm = MainView.getInstance().getAlgorithm();

        // check if passwords match
        if (confirmPassword && !password.equals(confirmedPassword)) {
            MainView.getInstance().print(
                resourceBundle.getString("PASSWORDS_DON_T_MATCH")
                );
            return;
        }

        Thread t = new Thread() {
            public void run() {
                try {
                    CipherFactory cf = new CipherFactoryImpl();
                    FileManager fm = new FileManagerImpl();
                    Cipher cipher = cf.createCipher(algorithm, password, mode);
                    fm.addObserver(progress);

                    if (infile.isDirectory()) {
                        if (outfile.exists() && !outfile.isDirectory()) {
                            MainView.getInstance().print(
                                resourceBundle.getString("INPUT_DIR_OUTPUT_FILE")
                                );
                            return;
                        }
                        if (outfile.equals(infile)) {
                            MainView.getInstance().print(
                                resourceBundle.getString("TARGET_EQUALS_SOURCE")
                                );
                            return;
                        }
                    }
                    //System.out.println(infile + " - " + outfile);

                    cryptDirectory(fm, infile, outfile, cipher);
                } catch (NoSuchAlgorithmException n) {
                    Logger.getLogger("org.dyndns.kwitte.smartcryp.ui")
                        .throwing("EncryptAction", "actionPerformed", n);
                    MainView.getInstance().print(
                        resourceBundle.getString("UNSUPPORTED_ALGORITHM"));
                } catch (IOException ioe) {
                    Logger.getLogger("org.dyndns.kwitte.smartcryp.ui")
                        .throwing("EncryptAction", "actionPerformed", ioe);
                    MainView.getInstance().print(
                        resourceBundle.getString("IO_ERROR")
                        + ioe.getMessage());
                } catch (InvalidKeyException i) {
                    Logger.getLogger("org.dyndns.kwitte.smartcryp.ui")
                        .throwing("EncryptAction", "actionPerformed", i);
                    MainView.getInstance().print(
                        resourceBundle.getString("INVALID_PASSWORD"));
                } finally {
                    EventQueue.invokeLater(
                        new Runnable() {
                            public void run() {
                                progress.setVisible(false);
                            }
                        }
                    );
                }
            }
        };
        t.setPriority(Thread.MIN_PRIORITY);
        t.start();
        progress.showProgress(resourceBundle.getString("PROGRESS_MESSAGE"));
    }

    private void cryptFile(FileManager fm, File infile, File outfile,
                           Cipher cipher) throws IOException {
        if (outfile.isDirectory()) {
            outfile = new File(outfile, infile.getName());
        }
        if (outfile.exists() && neverOverwriteExistingFile) {
            return;
        }
        if (outfile.exists() 
            && !overwriteAllWithoutConfirmation
            && !MainView.getInstance().cryptToStdout()) {
            int choice = new ConfirmDialog(
                MainView.getInstance(),
                resourceBundle.getString("CONFIRM_OVERWRITE") + outfile,
                resourceBundle.getString("CONFIRM_OVERWRITE_TITLE")
            ).showDialog();

            switch (choice) {
                case ConfirmDialog.YES_ALL_OPTION :
                    overwriteAllWithoutConfirmation = true;
                case ConfirmDialog.YES_OPTION :
                    break;
                case ConfirmDialog.NO_ALL_OPTION :
                    neverOverwriteExistingFile = true;
                case ConfirmDialog.NO_OPTION :
                    return;
                case ConfirmDialog.CANCEL_OPTION :
                    cancelled = true;
                    return;
                default : throw new AssertionError();
            }
        }

        OutputStream outstream =
            MainView.getInstance().cryptToStdout() ? 
                System.out : new FileOutputStream(outfile);
        fm.filterFile(infile,
                      outstream,
                      cipher);
    }

    private void cryptDirectory(FileManager fm, File sourceDir,
                                File targetDir, Cipher cipher)
            throws IOException {
        File[] fileList =
            new FileListImpl().listFiles(sourceDir, null);
        long totalFileSize = computeTotalFileSize(fileList);
        fm.setTotalSize(totalFileSize);

        for (int i = 0; i < fileList.length && !cancelled; i++) {
            File file = fileList[i];
            File absoluteOutFile =
                createTargetFile(file, sourceDir, targetDir);

            if (file.isDirectory()) {
                absoluteOutFile.mkdirs();
            }
            else {
                if (absoluteOutFile.getParent() != null) {
                    new File(absoluteOutFile.getParent()).mkdirs();
                }

                cryptFile(fm, file, absoluteOutFile, cipher);

            }
        }
    }

    // computes the name of the target file or directory
    private File createTargetFile(File file, File sourceDir, File targetDir) {
        assert file.getAbsolutePath().startsWith(
            sourceDir.getAbsolutePath());
        String relativeOutDirectoryName =
            file.getAbsolutePath().substring(
                    sourceDir.getAbsolutePath().length());
        if (!targetDir.toString().endsWith(FILE_SEPARATOR)
            && !relativeOutDirectoryName.startsWith(FILE_SEPARATOR)) {
            relativeOutDirectoryName =
            FILE_SEPARATOR + relativeOutDirectoryName;
        }
        String absoluteOutDirectoryName =
            targetDir + relativeOutDirectoryName;
        File absoluteTargetDirectory =
            new File(absoluteOutDirectoryName);
        return absoluteTargetDirectory;
    }

    private long computeTotalFileSize(File[] fileList) {
        long ret = 0;
        for (File f : fileList) {
            if (!f.isDirectory()) {
                ret += f.length();
            }
        }
        return ret;
    }
}
