/*
 * Smartcrypt - a tool to encrypt / decrypt files
 * Copyright (C) 2005 Kai Witte
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * For contact information see http://kwitte.dyndns.org/~kwitte/
 */
package org.dyndns.kwitte.smartcrypt.ui;

import org.dyndns.kwitte.smartcrypt.ui.controller.*;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.PBEParameterSpec;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import java.awt.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.ResourceBundle;
import java.util.logging.Logger;

/**
 * The main view for the gui version, and also the model when launched in GUI
 * mode. Methods must be called in the EDT unless documented otherwise.
 */
public final class MainView extends JFrame {
    private static final String VERSION = "2.4";
    private static final MainView INSTANCE = new MainView();

    private static final String DEFAULT_ALGORITHM = "PBEWithMD5AndDES";
    
    private ResourceBundle resourceBundle = ResourceBundle.getBundle(
        "org/dyndns/kwitte/smartcrypt/ui/mainview"
    );

    private JPasswordField passwordField1;
    private JPasswordField passwordField2;
    private JComboBox algorithmComboBox;
    private JButton sourceButton;
    private JPanel panel;
    private JButton outputButton;
    private JButton encryptButton;
    private JButton decryptButton;
    private JTextField sourceFileName;
    private JTextField outputFileName_;

    private Action infileAction = new SelectInfileAction();
    private Action outfileAction = new SelectOutfileAction();
    private Action encryptAction = new EncryptAction();
    private Action decryptAction = new DecryptAction();
    private JPanel inputFilePanel;
    private JComboBox outputFileName;
    
    private String[] targetDirectories = {
        System.getProperty("user.home"),
        System.getProperty("user.home")
            + System.getProperty("file.separator")
            + ".tiger",
        new File(".").getAbsoluteFile().toString()
    };
    private static final int TOOLTIP_DELAY = 15000;
    private JRadioButton radioButtonFile;
    private JRadioButton radioButton1;
    private JRadioButton radioButtonStdout;

    private MainView() {
        super("TigerCrypt version " + VERSION);
        registerListeners();

        super.setDefaultCloseOperation(EXIT_ON_CLOSE);
        super.setContentPane(panel);
        super.setSize(400, 400);
        super.pack();
    }

    private void registerListeners() {
        sourceButton.setAction(infileAction);
        outputButton.setAction(outfileAction);
        encryptButton.setAction(encryptAction);
        encryptButton.setMnemonic(
            resourceBundle.getString("ENCRYPT_BUTTON_MNEMONIC").charAt(0));
        decryptButton.setAction(decryptAction);
        decryptButton.setMnemonic(
            resourceBundle.getString("DECRYPT_BUTTON_MNEMONIC").charAt(0));

        DocumentListener dl = new DocumentListener() {
            public void insertUpdate(DocumentEvent e) { fileChanged(); }
            public void removeUpdate(DocumentEvent e) { fileChanged(); }
            public void changedUpdate(DocumentEvent e) { }
        };

        sourceFileName.addFocusListener(new SelectAllListener());
        outputFileName.getEditor().getEditorComponent().
            addFocusListener(new SelectAllListener());
        passwordField1.addFocusListener(new SelectAllListener());
        passwordField2.addFocusListener(new SelectAllListener());
        algorithmComboBox.getEditor().getEditorComponent().
            addFocusListener(new SelectAllListener());
        sourceFileName.getDocument().addDocumentListener(dl);
        outputFileName.addItemListener(
            new ItemListener() {
                public void itemStateChanged(ItemEvent e) {
                    fileChanged();
                }
            }
        );
        for (String item : targetDirectories) {
            outputFileName.addItem(item);
        }
//        outputFileName.addKeyListener(
//            new KeyAdapter() {
//                @Override public void keyTyped(KeyEvent e) {
//                    fileChanged();
//                }
//            }
//        );
        //outputFileName.getDocument().addDocumentListener(dl);

        for (String algorithm : supportedAlgorithms()) {
            algorithmComboBox.addItem(algorithm);
        }

        // enable dropping of files
        inputFilePanel.setTransferHandler(
            new org.dyndns.kwitte.smartcrypt.ui.controller.DropFileHandler() {
                public void dropFiles(File[] f) {
                    if (f.length != 0) {
                        setInfile(f[0]);
                    }
                }
            }
        );
        sourceFileName.setTransferHandler(
            new DropFileHandler(sourceFileName.getTransferHandler()) {
                public void dropFiles(File[] f) {
                    if (f.length != 0) {
                        setInfile(f[0]);
                    }
                }
            }
        );
        
        // tooltip
        String tooltip = MessageFormat.format(
                resourceBundle.getString("FILE_PANEL_TOOLTIP"), 
                MainView.class.getResource("/res/Open16.gif").toString());
        //System.out.println(tooltip);
        ToolTipManager.sharedInstance().setDismissDelay(TOOLTIP_DELAY);
        inputFilePanel.setToolTipText(tooltip);
        sourceFileName.setToolTipText(tooltip);
        sourceButton.setToolTipText(tooltip);
        
        // readiobuttons
        ButtonGroup buttonGroup = new ButtonGroup();
        buttonGroup.add(radioButtonFile);
        buttonGroup.add(radioButtonStdout);
        radioButtonFile.setMnemonic(
            resourceBundle.getString("OUTPUT_FILE_MNEMONIC").charAt(0));
        radioButtonStdout.setMnemonic(
            resourceBundle.getString("OUTPUT_STDOUT_MNEMONIC").charAt(0));
    }
    
    public boolean cryptToStdout() {
        return radioButtonStdout.isSelected();
    }
    
    /**
     * Returns the instance of this singleton. Thread-safe.
     * @return the instance of this singleton
     */
    public static MainView getInstance() {
        return INSTANCE;
    }

    public String getString(String key) {
        return resourceBundle.getString(key);
    }

    private void fileChanged() {
        boolean canCrypt = 
            sourceFileName.getText().length() != 0; 
            //&& outputFileName.getSelectedItem().toString().length() != 0;
        encryptAction.setEnabled(canCrypt);
        decryptAction.setEnabled(canCrypt);
    }

    public void setInfile(File f) {
        sourceFileName.setText(f.toString());
//        encryptAction.setEnabled(outputFileName.getSelectedItem().toString().length() != 0);
//        decryptAction.setEnabled(outputFileName.getSelectedItem().toString().length() != 0);
    }

    public void setOutfile(File f) {
        outputFileName.addItem(f.toString());
        outputFileName.setSelectedItem(f.toString());
        encryptAction.setEnabled(sourceFileName.getText().length() != 0);
        decryptAction.setEnabled(sourceFileName.getText().length() != 0);
    }

    public String getAlgorithm() {
        return (String) algorithmComboBox.getSelectedItem();
    }

    public String getInfile() {
        return sourceFileName.getText();
    }

    public String getOutfile() {
        return (String) outputFileName.getSelectedItem();
    }

    public String getPassword() {
        return new String(passwordField1.getPassword());
    }

    public String getPasswordConfirm() {
        return new String(passwordField2.getPassword());
    }

    //@todo better solution to find algorithms
    private String[] supportedAlgorithms() {
        List<String> ret = new ArrayList<String>();

        // just try what works (without exception)
        try {
            final String ALGORITHM = "PBEWithMD5AndTripleDES";
            final String PASSWORD = "asdfasdf";
            final byte[] SALT = new byte[] { 3, 4, 27, 9, 4, 9, 27, 100 };

            SecretKeyFactory pfac =
                SecretKeyFactory.getInstance(ALGORITHM);
            Key key = pfac.generateSecret(new PBEKeySpec(PASSWORD.toCharArray()));
            PBEParameterSpec pspec = new PBEParameterSpec(SALT, 1024);
            AlgorithmParameters params = null;
                params = AlgorithmParameters.getInstance("PBE");
            params.init(pspec);
            Cipher cipher = Cipher.getInstance(ALGORITHM);

            // The exception occurs here if TripleDES is not supported
            cipher.init(Cipher.ENCRYPT_MODE, key, params);

            ret.add(ALGORITHM);
        } catch (InvalidKeyException ike) {
            Logger.getLogger("org.dyndns.kwitte.smartcrypt").
                throwing("MainView", "supportedAlgorithms", ike);
        } catch (NoSuchAlgorithmException i) {
            Logger.getLogger("org.dyndns.kwitte.smartcrypt").
                throwing("MainView", "supportedAlgorithms", i);
        } catch (InvalidAlgorithmParameterException i) {
            throw new AssertionError(i);
        } catch (InvalidKeySpecException i) {
            throw new AssertionError(i);
        } catch (InvalidParameterSpecException i) {
            throw new AssertionError(i);
        } catch (NoSuchPaddingException n) {
            throw new AssertionError(n);
        }

        ret.add(DEFAULT_ALGORITHM);

        return ret.toArray(new String[] {});
    }

    /**
     * Shows the message in a dialog.
     * The user has to confirm it before he can do anything else.
     * May be called from any thread. It does not return before
     * the user confirmed it.
     * @param message the message
     */
    public void print(final String message) {
        Runnable printMessage = new Runnable() {
            public void run() {
                JOptionPane.showMessageDialog(
                    MainView.getInstance(),
                    message,
                    getString("DEFAULT_MESSAGE_TITLE"),
                    JOptionPane.ERROR_MESSAGE);
            }
        };

        if (SwingUtilities.isEventDispatchThread()) {
            printMessage.run();
        }

        else {
            try {
                SwingUtilities.invokeAndWait(printMessage);
            } catch (InterruptedException ie) {
                // shouldn't be possible
                throw new AssertionError(ie);
            } catch (java.lang.reflect.InvocationTargetException ite) {
                // shouldn't be possible
                throw new AssertionError(ite);
            }
        }
    }

    public void setData(MainModel data) {
        sourceFileName.setText(data.getSourceFileName());
        passwordField1.setText(data.getPassword1());
        passwordField2.setText(data.getPassword2());
    }

    public void getData(MainModel data) {
        data.setSourceFileName(sourceFileName.getText());
        data.setPassword1(passwordField1.getText());
        data.setPassword2(passwordField2.getText());
    }

    {
// GUI initializer generated by IntelliJ IDEA GUI Designer 
// >>> IMPORTANT!! <<<
// DO NOT EDIT OR ADD ANY CODE HERE!
        $$$setupUI$$$();
    }

    /**
     * Method generated by IntelliJ IDEA GUI Designer
     * >>> IMPORTANT!! <<<
     * DO NOT edit this method OR call it in your code!
     */
    private void $$$setupUI$$$() {
        panel = new JPanel();
        panel.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(5, 1,
                                                                           new Insets(
                                                                               0,
                                                                               0,
                                                                               0,
                                                                               0),
                                                                           -1,
                                                                           -1));
        final JPanel panel1 = new JPanel();
        panel1.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(1,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel.add(panel1, new com.intellij.uiDesigner.core.GridConstraints(0, 0,
                                                                           1, 1,
                                                                           com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                           com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           null,
                                                                           null,
                                                                           null));
        final JPanel panel2 = new JPanel();
        panel2.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(2,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel1.add(panel2, new com.intellij.uiDesigner.core.GridConstraints(0,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            null,
                                                                            null,
                                                                            null));
        final JPanel panel3 = new JPanel();
        panel3.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(2,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel2.add(panel3, new com.intellij.uiDesigner.core.GridConstraints(0,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            null,
                                                                            null,
                                                                            null));
        final JPanel panel4 = new JPanel();
        panel4.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(2,
                                                                            3,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel3.add(panel4, new com.intellij.uiDesigner.core.GridConstraints(1,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            null,
                                                                            null,
                                                                            null));
        panel4.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            com.intellij.uiDesigner.core.SupportCode.getResourceString(
                "org/dyndns/kwitte/smartcrypt/ui/mainview",
                "OUTPUT_BORDER_TITLE")));
        outputButton = new JButton();
        outputButton.setEnabled(true);
        outputButton.setText("select file");
        panel4.add(outputButton,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 2, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                    1, 1, null,
                                                                    null,
                                                                    null));
        outputFileName = new JComboBox();
        outputFileName.setEditable(true);
        panel4.add(outputFileName,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 1, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_HORIZONTAL,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null, null,
                                                                    null));
        radioButtonFile = new JRadioButton();
        radioButtonFile.setSelected(true);
        com.intellij.uiDesigner.core.SupportCode.setTextFromBundle(
            radioButtonFile, "org/dyndns/kwitte/smartcrypt/ui/mainview",
            "OUTPUT_FILE");
        panel4.add(radioButtonFile,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                    com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_SHRINK
                                                                    | com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null, null,
                                                                    null));
        radioButtonStdout = new JRadioButton();
        radioButtonStdout.setEnabled(true);
        com.intellij.uiDesigner.core.SupportCode.setTextFromBundle(
            radioButtonStdout, "org/dyndns/kwitte/smartcrypt/ui/mainview",
            "OUTPUT_STDOUT");
        panel4.add(radioButtonStdout,
                   new com.intellij.uiDesigner.core.GridConstraints(1, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                    com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_SHRINK
                                                                    | com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null, null,
                                                                    null));
        inputFilePanel = new JPanel();
        inputFilePanel.setLayout(
            new com.intellij.uiDesigner.core.GridLayoutManager(1, 2, new Insets(
                0, 0, 0, 0), -1, -1));
        panel3.add(inputFilePanel,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                    com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_SHRINK
                                                                    | com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_SHRINK
                                                                    | com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_GROW,
                                                                    null, null,
                                                                    null));
        inputFilePanel.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            com.intellij.uiDesigner.core.SupportCode.getResourceString(
                "org/dyndns/kwitte/smartcrypt/ui/mainview",
                "SOURCE_BORDER_TITLE")));
        sourceButton = new JButton();
        sourceButton.setRequestFocusEnabled(true);
        sourceButton.setRolloverEnabled(false);
        sourceButton.setText("select file");
        inputFilePanel.add(sourceButton,
                           new com.intellij.uiDesigner.core.GridConstraints(0,
                                                                            1,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                            1,
                                                                            1,
                                                                            null,
                                                                            null,
                                                                            null));
        sourceFileName = new JTextField();
        sourceFileName.setText("");
        inputFilePanel.add(sourceFileName,
                           new com.intellij.uiDesigner.core.GridConstraints(0,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_HORIZONTAL,
                                                                            com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_WANT_GROW,
                                                                            com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                            null,
                                                                            new Dimension(
                                                                                150,
                                                                                -1),
                                                                            null));
        final JPanel panel5 = new JPanel();
        panel5.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(1,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel2.add(panel5, new com.intellij.uiDesigner.core.GridConstraints(1,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                            com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_SHRINK
                                                                            | com.intellij.uiDesigner.core.GridConstraints
                                                                                .SIZEPOLICY_CAN_GROW,
                                                                            null,
                                                                            null,
                                                                            null));
        panel5.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            com.intellij.uiDesigner.core.SupportCode.getResourceString(
                "org/dyndns/kwitte/smartcrypt/ui/mainview",
                "ALGORITHM_BORDER_TITLE")));
        algorithmComboBox = new JComboBox();
        algorithmComboBox.setEditable(true);
        panel5.add(algorithmComboBox,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_HORIZONTAL,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null, null,
                                                                    null));
        final JPanel panel6 = new JPanel();
        panel6.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(1,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel6.setFocusable(false);
        panel6.setName("");
        panel.add(panel6, new com.intellij.uiDesigner.core.GridConstraints(1, 0,
                                                                           1, 1,
                                                                           com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                           com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           null,
                                                                           null,
                                                                           null));
        panel6.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            com.intellij.uiDesigner.core.SupportCode.getResourceString(
                "org/dyndns/kwitte/smartcrypt/ui/mainview",
                "PASSPHRASE_BORDER_TITLE")));
        passwordField1 = new JPasswordField();
        panel6.add(passwordField1,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_HORIZONTAL,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_WANT_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null,
                                                                    new Dimension(
                                                                        150,
                                                                        -1),
                                                                    null));
        decryptButton = new JButton();
        decryptButton.setEnabled(false);
        decryptButton.setRequestFocusEnabled(true);
        decryptButton.setRolloverEnabled(false);
        decryptButton.setText("decrypt");
        decryptButton.setToolTipText("");
        panel.add(decryptButton,
                  new com.intellij.uiDesigner.core.GridConstraints(2, 0, 1, 1,
                                                                   com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                   com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                   com.intellij.uiDesigner.core.GridConstraints
                                                                       .SIZEPOLICY_CAN_SHRINK
                                                                   | com.intellij.uiDesigner.core.GridConstraints
                                                                       .SIZEPOLICY_CAN_GROW,
                                                                   com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                   null, null,
                                                                   null));
        final JPanel panel7 = new JPanel();
        panel7.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(1,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel.add(panel7, new com.intellij.uiDesigner.core.GridConstraints(3, 0,
                                                                           1, 1,
                                                                           com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                           com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           null,
                                                                           null,
                                                                           null));
        panel7.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            com.intellij.uiDesigner.core.SupportCode.getResourceString(
                "org/dyndns/kwitte/smartcrypt/ui/mainview",
                "PASSPHRASE_BORDER_TITLE_CONFIRM")));
        passwordField2 = new JPasswordField();
        panel7.add(passwordField2,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_WEST,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_HORIZONTAL,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_WANT_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null,
                                                                    new Dimension(
                                                                        150,
                                                                        -1),
                                                                    null));
        final JPanel panel8 = new JPanel();
        panel8.setLayout(new com.intellij.uiDesigner.core.GridLayoutManager(1,
                                                                            1,
                                                                            new Insets(
                                                                                0,
                                                                                0,
                                                                                0,
                                                                                0),
                                                                            -1,
                                                                            -1));
        panel.add(panel8, new com.intellij.uiDesigner.core.GridConstraints(4, 0,
                                                                           1, 1,
                                                                           com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                           com.intellij.uiDesigner.core.GridConstraints.FILL_BOTH,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_SHRINK
                                                                           | com.intellij.uiDesigner.core.GridConstraints
                                                                               .SIZEPOLICY_CAN_GROW,
                                                                           null,
                                                                           null,
                                                                           null));
        encryptButton = new JButton();
        encryptButton.setEnabled(false);
        encryptButton.setText("encrypt");
        encryptButton.setToolTipText("");
        panel8.add(encryptButton,
                   new com.intellij.uiDesigner.core.GridConstraints(0, 0, 1, 1,
                                                                    com.intellij.uiDesigner.core.GridConstraints.ANCHOR_CENTER,
                                                                    com.intellij.uiDesigner.core.GridConstraints.FILL_NONE,
                                                                    com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_SHRINK
                                                                    | com.intellij.uiDesigner.core.GridConstraints
                                                                        .SIZEPOLICY_CAN_GROW,
                                                                    com.intellij.uiDesigner.core.GridConstraints.SIZEPOLICY_FIXED,
                                                                    null, null,
                                                                    null));
    }
}
