package org.dyndns.kwitte.smartcrypt.ui;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.ResourceBundle;

/**
 * A confirmation dialog with the options 'yes', 'no', 'yes to all',
 * 'no to all' and 'cancel'. Must be created in the EDT, and all methods must
 * be called in the EDT.
 */
public final class ConfirmDialog {
    private static final ResourceBundle resourceBundle =
        ResourceBundle.getBundle(
            "org/dyndns/kwitte/smartcrypt/ui/confirmdialog"
        );
    private static final Dimension INITIAL_SIZE = new Dimension(400, 350);

    /**
     * Indicates that 'yes' has been chosen.
     * @see #showDialog()
     */
    public static final int YES_OPTION = 2727;

    /**
     * Indicates that 'no' has been chosen.
     * @see #showDialog()
     */
    public static final int NO_OPTION = 2728;

    /**
     * Indicates that 'yes to all' has been chosen.
     * @see #showDialog()
     */
    public static final int YES_ALL_OPTION = 2730;

    /**
     * Indicates that 'no to all' has been chosen.
     * @see #showDialog()
     */
    public static final int NO_ALL_OPTION = 2731;
    
    /**
     * Indicates that 'cancel' has been chosen.
     * @see #showDialog()
     */
    public static final int CANCEL_OPTION = 2729;

    private static final String[] BUTTON_LABELS =
        {resourceBundle.getString("YES_OPTION"),
         resourceBundle.getString("NO_OPTION"),
         resourceBundle.getString("YES_ALL_OPTION"),
         resourceBundle.getString("NO_ALL_OPTION"),
         resourceBundle.getString("CANCEL_OPTION")};
    private static final int[] OPTIONS =
        {YES_OPTION,
         NO_OPTION,
         YES_ALL_OPTION,
         NO_ALL_OPTION,
         CANCEL_OPTION};
    private static final char[] MNEMONICS =
        {resourceBundle.getString("YES_OPTION_MNEMONIC").charAt(0),
         resourceBundle.getString("NO_OPTION_MNEMONIC").charAt(0),
         resourceBundle.getString("YES_ALL_OPTION_MNEMONIC").charAt(0),
         resourceBundle.getString("NO_ALL_OPTION_MNEMONIC").charAt(0),
         resourceBundle.getString("CANCEL_OPTION_MNEMONIC").charAt(0)};

    private static final int DEFAULT_BUTTON = 1;

    private int value;
    private final JDialog dialog;

    /**
     * Creates a new instance.
     * @param owner the owner of this dialog
     * @param message the message to be displayed
     * @param title the title of the dialog window
     */
    public ConfirmDialog(Frame owner, String message, String title) {
        dialog = new JDialog(owner, title, true);
        //dialog.setSize(INITIAL_SIZE);

        // buttons
        JButton[] buttons = createButtons();
        dialog.getRootPane().setDefaultButton(buttons[DEFAULT_BUTTON]);
        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.LINE_AXIS));
        buttonPane.setBorder(BorderFactory.createEmptyBorder(0, 10, 10, 10));
        for (JButton button : buttons) {
            buttonPane.add(Box.createHorizontalGlue());
            buttonPane.add(button);
            buttonPane.add(Box.createRigidArea(new Dimension(10, 0)));
        }

        // text
        JPanel textPane = new JPanel();
        textPane.setLayout(new BoxLayout(textPane, BoxLayout.PAGE_AXIS));
        JLabel label = new JLabel(message);
        textPane.add(label);
        textPane.add(Box.createRigidArea(new Dimension(0,5)));
        textPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

        dialog.getContentPane().add(textPane, BorderLayout.CENTER);
        dialog.getContentPane().add(buttonPane, BorderLayout.PAGE_END);

        dialog.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                assert value == 0;
                value = CANCEL_OPTION;
            }
        });

        dialog.pack();
    }

    //Create and initialize the buttons.
    private JButton[] createButtons() {
        assert OPTIONS.length == MNEMONICS.length;
        assert OPTIONS.length == BUTTON_LABELS.length;

        JButton[] buttons = new JButton[OPTIONS.length];

        for (int i = 0; i < OPTIONS.length; i++) {
            final String label = BUTTON_LABELS[i];
            final int option = OPTIONS[i];
            final char mnemonic = MNEMONICS[i];
            assert label.toLowerCase().indexOf(mnemonic) != -1;

            JButton button = new JButton(
                new AbstractAction() {
                    {
                        putValue(NAME, label);
                    }
                    public void actionPerformed(ActionEvent e) {
                        value = option;
                        dialog.setVisible(false);
                    }
                }
            );
            button.setMnemonic(mnemonic);
            buttons[i] = button;
        }

        return buttons;
    }

    /**
     * Shows the dialog.
     * Returns when this dialog is closed.
     * @return {@link YES_OPTION},
     *         {@link  NO_OPTION},
     *         {@link  YES_ALL_OPTION},
     *         {@link  NO_ALL_OPTION} or
     *         {@link  CANCEL_OPTION}, depending on the user's choice
     */
    public int showDialog() {
        value = 0;
        dialog.setVisible(true);
        return value;
    }

    /**
     * A main method to test the dialog.
     * @param args not used
     */
    public static void main(String[] args) {
        SwingUtilities.invokeLater(
            new Runnable() {
                public void run() {
                    System.out.println(
                        new ConfirmDialog(null, "test", "test").showDialog());
                    for (int i = 0; i < OPTIONS.length; i++) {
                        System.out.println(BUTTON_LABELS[i] + " : " + OPTIONS[i]);
                    }
                    System.exit(0);
                }
            }
        );
    }
}
