/*
 * Smartcrypt - a tool to encrypt / decrypt files
 * Copyright (C) 2005 Kai Witte
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * For contact information see http://kwitte.dyndns.org/~kwitte/
 */
package org.dyndns.kwitte.smartcrypt;

import javax.crypto.*;
import java.io.*;
import java.util.logging.*;

public class FileManagerImpl extends FileManager {
    private static final int EOF = -1;
    private static final Logger logger =
        Logger.getLogger("org.dyndns.kwitte.smartcrypt");
    private static final int MAX_PROGRESS = 100;
    private static final int MIN_STEPS = 1;

    private long numFiles;
    private long totalSize;
    private long lastPosition;
    private static final int BUFFERSIZE = 1024;

    /**
     * Creates a new instance.
     */
    public FileManagerImpl() {
    }

    /**
     * Creates a new instance.
     * @param numFiles the number of files, used for progress calculation
     * @param totalSize the sum of the size of all files, 
     * used for progress calculation
     */
    public FileManagerImpl(long numFiles, long totalSize) {
        this.numFiles = numFiles;
        this.totalSize = totalSize;
    }

    public void filterFile(File infile, OutputStream outfile,
                           Cipher incipher)
            throws IOException {
        if (infile == null || outfile == null || incipher == null) {
            throw new NullPointerException();
        }

        InputStream in = new BufferedInputStream(new CipherInputStream(
                             new FileInputStream(infile), incipher));
        OutputStream out;
        if (EncryptDecrypt.cryptToStdOut()) {
            out = new BufferedOutputStream(System.out);
        }
        else {
            out = new BufferedOutputStream(outfile);
        }

        // computations for progress notifications
        logger.log(Level.FINE, "starting file creation");
        long endPosition = totalSize == 0 ? infile.length() : totalSize;
        long currentPosition = totalSize == 0 ? 0 : lastPosition;
        final long stepsBetweenProgressMessages =
             (endPosition / MAX_PROGRESS / BUFFERSIZE) + MIN_STEPS;
        assert stepsBetweenProgressMessages > 0;

        try {
            byte[] buffer = new byte[BUFFERSIZE];

            for (int i = 0; i != EOF; ) {
                for (long j = 0;
                     j < stepsBetweenProgressMessages
                     &&  (i = in.read(buffer)) != EOF;
                     j++) {
                    currentPosition += i;
                    out.write(buffer, 0, i);
                }

                // progress notification
                long relativeProgress =
                    MAX_PROGRESS * currentPosition / (endPosition + MIN_STEPS);
                if (relativeProgress < MAX_PROGRESS) {
                    setChanged();
                    notifyObservers((int) relativeProgress);
                }
            }
        } finally {
            lastPosition = currentPosition;
            out.close();
            in.close();
        }
    }

    public void setTotalSize(long totalFileSize) {
        this.totalSize = totalFileSize;
    }
}
