/*
 * Smartcrypt - a tool to encrypt / decrypt files
 * Copyright (C) 2005 Kai Witte
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * For contact information see http://kwitte.dyndns.org/~kwitte/
 */


package org.dyndns.kwitte.smartcrypt;

import org.dyndns.kwitte.smartcrypt.ui.MainView;

import javax.crypto.*;
import javax.swing.*;
import java.security.*;
import java.io.IOException;
import java.io.File;
import java.io.FileOutputStream;
import java.util.logging.Logger;
import java.util.ResourceBundle;
import java.util.Locale;

import static java.lang.System.err;

/**
 * The main class.
 */
public final class EncryptDecrypt {
    private static final String DECRYPT_FLAG = "-d(s)?";

    private static final int OPTIONS = 0;
    private static final int INFILE = 1;
    private static final int OUTFILE = 2;
    private static final int PASSWORD = 3;
    private static final int ALGORITHM = 4;
    private static volatile boolean cryptToStdOut;

    public static boolean cryptToStdOut() {
        return cryptToStdOut;
    }

    private static final ResourceBundle resourceBundle =
        ResourceBundle.getBundle(
            "org/dyndns/kwitte/smartcrypt/console"
        );

    private static void usage() {
        err.println(resourceBundle.getString("USAGE"));
    }

    /**
     * The main method.
     * @param args the arguments:<br>
     * -d to decrypt (optional, defaults to encrypt mode)<br>
     * infile<br>
     * outfile<br>
     * password<br>
     * algorithm<br>
     * With no arguments the GUI is launched.
     */
    public static void main(String[] args) {
        //System.out.println("main");
        if (args.length == 5 && args[OPTIONS].matches(DECRYPT_FLAG)) {
            //System.out.println("match 5");
            if (args[OPTIONS].contains("s")) {
                //System.out.println("match s");
                cryptToStdOut = true;
                main(new String[] {
                        args[INFILE],
                        args[OUTFILE],
                        args[PASSWORD],
                        args[ALGORITHM]},
                     Cipher.DECRYPT_MODE);
            }
            else {
                main(new String[] {
                        args[INFILE],
                        args[OUTFILE],
                        args[PASSWORD],
                        args[ALGORITHM]},
                     Cipher.DECRYPT_MODE);
            }
        }
        else if (args.length == 0) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    MainView.getInstance().setVisible(true);
                }
            });
        }
        else {
            main(args, Cipher.ENCRYPT_MODE);
        }
    }

    public static void main(String[] args, int mode) {
        if (!(args.length == 4)) {
            usage();
            return;
        }

        String infile = args[0];
        String outfile = args[1];
        String password = args[2];
        String algorithm = args[3];

        CipherFactory cf = new CipherFactoryImpl();
        FileManager fm = new FileManagerImpl();
        try {
            fm.filterFile(new File(infile), new FileOutputStream(outfile),
                cf.createCipher(algorithm, password, mode));
        } catch (NoSuchAlgorithmException n) {
            Logger.getLogger("org.dyndns.kwitte.smartcrypt").throwing(
                "EncryptDecrypt", "main", n);
            err.println(resourceBundle.getString("UNSUPPORTED_ALGORITHM")
                        + algorithm);
            usage();
        } catch (IOException ioe) {
            Logger.getLogger("org.dyndns.kwitte.smartcrypt").throwing(
                "EncryptDecrypt", "main", ioe);
            err.println(resourceBundle.getString("IO_ERROR") 
                               + ioe.getMessage());
        } catch (InvalidKeyException i) {
            Logger.getLogger("org.dyndns.kwitte.smartcrypt").throwing(
                "EncryptDecrypt", "main", i);
            err.println(resourceBundle.getString("INVALID_PASSPHRASE")
                        + password + " - " + algorithm);
            err.println(i.getMessage());
        }
    }
}
